#!/usr/bin/env python


# xbmcsender.py
# mouser@donationcoder.com
# version 1.1 - 12/26/13
# Commandline utility that will send urls to a remote xbmc machine for playback
# This includes arbitrary urls, and special handling for youtube and vimeo urls via plugin.
# It will also automatically ftp local files and then cause them to be played -- this can be used to show images or uploade mp3/avi/wmv/etc files
# Useful as a helper for drag+drop operations to quickly send a url or file to your xbmc machine.



try:
    import json
except ImportError:
    import simplejson as json
import urllib2, base64, time, socket, struct

import re, os, sys



class XBMCJSON:


    def __init__(self, settings):
        self.settings = settings
        #
        self.server = 'http://%s:%s/jsonrpc' % (self.getset('hostname'), self.getset('port'))
        #
        self.version = '2.0'

    def __call__(self, **kwargs):
        method = '.'.join(map(str, self.n))
        self.n = []
        return XBMCJSON.__dict__['Request'](self, method, kwargs)

    def __getattr__(self,name):
        if not self.__dict__.has_key('n'):
            self.n=[]
        self.n.append(name)
        return self

    def Request(self, method, kwargs):
        data = [{}]
        data[0]['method'] = method
        data[0]['params'] = kwargs
        data[0]['jsonrpc'] = self.version
        data[0]['id'] = 1

        data = json.JSONEncoder().encode(data)
        content_length = len(data)

        content = {
            'Content-Type': 'application/json',
            'Content-Length': content_length,
        }

        request = urllib2.Request(self.server, data, content)
        username = self.getset('username')
        password = self.getset('password')
        base64string = base64.encodestring('%s:%s' % (username, password)).replace('\n', '')
        request.add_header("Authorization", "Basic %s" % base64string)

        f = urllib2.urlopen(request)
        response = f.read()
        f.close()
        response = json.JSONDecoder().decode(response)

        try:
            return response[0]['result']
        except:
            return response[0]['error']



    def getset(self, keyname, defaultval='__ERROR__'):
        if (keyname in self.settings):
            return self.settings[keyname]
        if (defaultval == '__ERROR__'):
            raise Exception('Expected setting key not found: ' + keyname)
        return defaultval

    def setsetting(self, keyname, val):
        self.settings[keyname]=val

    def setsettings(self, vals):
        self.settings.update(vals)



    def wakeup(self):

        addr_byte = self.self.getset('mac_address').split(':')
        hw_addr = struct.pack('BBBBBB',
            int(addr_byte[0], 16),
            int(addr_byte[1], 16),
            int(addr_byte[2], 16),
            int(addr_byte[3], 16),
            int(addr_byte[4], 16),
            int(addr_byte[5], 16)
        )
        #Send wakeup packet
        msg = '\xff' * 6 + hw_addr * 16
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
        s.sendto(msg, ("255.255.255.255", 9))







    # see https://gist.github.com/dataolle/4207390
    def playuri_smart(self, uri):
        # smart play a url based on what it is

        # is it a local file that needs uploading?
        localfilere = re.search(r"[a-zA-Z]\:\\",uri)
        if (not localfilere):
            localfilere = re.search(r"^\/.*",uri)
        if (localfilere):
            return self.playuri_localfile(uri)

        # is it youtube? if so ask appropriate plugin to handle it
        youtubere = re.search(r".*youtube.*v\=([a-zA-Z0-9\-_]+)[^/]*$",uri)
        if (not youtubere):
            youtubere = re.search(r".*youtube.*v\=([a-zA-Z0-9]+)[^/]*$",uri)
        if (youtubere):
            vidid = youtubere.group(1)
            return self.playuri_youtube(vidid)

        # is it vimeo? if so ask appropriateplugin to handle it
        vimeore = re.search(r".*vimeo\.com\/([a-zA-Z0-9]+)[^/]*$",uri)
        if (not vimeore):
            vimeore = re.search(r".*vimeo\.com\.*/video/([a-zA-Z0-9]+)[^/]*$",uri)
        if (vimeore):
            vidid = vimeore.group(1)
            return self.playuri_vimeo(vidid)

        else:
            # fall back on file
            return self.playuri_urlasfile(uri)



    def playuri_urlasfile(self, uri):
        # play catchall fallback url
        print('Attempting to play generic link: ' + uri + '.')
        method = "Player.Open"
        kwargs = { "item" : { "file": uri } }
        return self.Request(method, kwargs)


    def playuri_youtube(self, vidid):
        # play youtube url
        print('Attempting to play YOUTUBE link, videoid: ' + vidid + '.')
        method = "Player.Open"
        url = "plugin://plugin.video.youtube/?action=play_video&videoid=" + vidid
        kwargs = { "item" : { "file": url } }
        return self.Request(method, kwargs)


    def playuri_vimeo(self, vidid):
        # play vimeo url
        print('Attempting to play VIMEO link, videoid: ' + vidid + '.')
        method = "Player.Open"
        url = "plugin://plugin.video.vimeo/?action=play_video&videoid=" + vidid
        kwargs = { "item" : { "file": url } }
        return self.Request(method, kwargs)


    def playuri_localfile(self, path):
        # this is tricky -- the file is on OUR pc and we want to move it to the xbmc pc and play it!
        # first we upload it!
        result = self.ftp_upload(path)
        # and now we ask xbmc to play it from newly uploaded location
        if (True):
            remoteuri = result["remoteuri"]
            print(' Attempting to play newly uploaded file "' + remoteuri + '".')
            return self.playuri_urlasfile(remoteuri)
        return False






    def ftp_upload(self, localfilepath, targetfolder=''):
        import ftplib
        from ftplib import FTP
        #
        remotefname = os.path.basename(localfilepath)
        #
        ftp_host = self.self.getset('ftp_host');
        ftp_port = self.self.getset('ftp_port','');
        ftp_username = self.self.getset('ftp_username');
        ftp_password = self.self.getset('ftp_password');
        targetfolder = self.getset('ftp_targetfolder',targetfolder)
        ftp_remotepath = self.getset('ftp_remotepath')

        # connect
        ftpobj = FTP()
        if (ftp_port==None or ftp_port==''):
            print('Connecting to ftp server ' + ftp_host + '.')
            session = ftpobj.connect(ftp_host)
        else:
            print('Connecting to ftp server ' + ftp_host + ' (port' + ftp_port + ').')
            session = ftpobj.connect(ftp_host, ftp_port)
        # login
        print(' Connected.  Logging in as user "' + ftp_username + '".')
        ftpobj.login(ftp_username, ftp_password)
        # change to targetflder
        if (targetfolder != ''):
             print(' Changing to remote directory "' + targetfolder + '".')
             ftpobj.cwd(targetfolder)
        # ok upload file
        file = open(localfilepath,'rb')
        print(' Uploading file "' + localfilepath + '" as "' + remotefname + '".')
        ftpresult = ftpobj.storbinary('STOR ' + remotefname, file)
        # close file
        file.close()
        # success
        remoteuri = ftp_remotepath + "/" + targetfolder + "/" + remotefname
        result = {
            "ftpresult" : ftpresult,
            "remoteuri" : remoteuri,
            }
        return result




    @classmethod
    def handle_main(classobj, settings, argv=None):
        # defaults
        confirmexit = True
        uri = None

        # default argv
        if argv is None:
            argv = sys.argv
        # process args simply (no use of arg library because we are doing this so simply and dont want requirements)
        for anarg in argv[1:]:
            if (anarg==""):
                contunue;
            if (anarg[0]=='-'):
                # option
                option = anarg
                if (option=="--exit"):
                    confirmexit = False
                elif (option=="--help"):
                    classobj.showhelp()
                elif (option=="--debug"):
                    print "Settings: "+str(settings)
                else:
                    raise("Unknown commandline option '{0}'.".format(option))
            else:
                uri = anarg

        # require commandline arg
        if (uri == None):
            # error OR try to get from clipboard
            classobj.showhelp()
            sys.exit('ERROR: You must specify a uri or file path on commandline.')
            #uri = classobj.get_clipboardtext()
            #print ('No uri passed on commandline, using clipboard contents (' + uri +').')

        # debug?
        #print "Settings: "+str(settings)

        # create the xbmc object
        xbmc = XBMCJSON(settings)

        # invoke it
        result = xbmc.playuri_smart(uri)

        # result
        print "Returned from request: " + str(result)

        if (confirmexit):
            raw_input("Press Enter to exit...")


    @classmethod
    def showhelp(classobj):
        print "xbmcsender commandline help:"
        print "Pass url (uri) on commandline, enclosed in double quotes."
        print "You can also pass the option --exit to exit without confirmation."


    @classmethod
    def get_clipboardtext(classobj):
        import win32clipboard
        win32clipboard.OpenClipboard()
        data = win32clipboard.GetClipboardData()
        win32clipboard.CloseClipboard()
        return data







    def internal_tests():
        # internal tests that will only work on my pc
        # ftp test
        localfilepath = 'E:\MyDocs\MultimediaFiles\MyPictures\Found\watercolor.jpg'
        #result = xbmc.ftp_upload(localfilepath)
        result = xbmc.playuri_localfile(localfilepath)
        print "Returned from request: " + str(result)
        # sample url
        #uri = 'http://crooksandliars.com/medialoader/27976/ae605/wmv/TOF-Media-Kurtz-091813.wmv'
        #uri = 'http://www.youtube.com/watch?feature=player_detailpage&v=961dZHVltA4'
        uri = 'https://vimeo.com/67069182'
        result = xbmc.playuri_smart(uri)
        print "Returned from request: " + str(result)







# run main func when invoked from shell
if __name__ == "__main__":
    from xbmcsettings import settings
    XBMCJSON.handle_main(settings)


